#include "goose-publisher-app.h"
#include "ns3/simulator.h"
#include "ns3/goose_publisher.h"
#include "ns3/uinteger.h"

ns3::GOOSEPublisher::
GOOSEPublisher()
{
    NS_LOG_FUNCTION(this);
}

ns3::TypeId
ns3::GOOSEPublisher::GetTypeId()
{
    static TypeId tid =
    TypeId("ns3::GOOSEPublisher")
        .SetParent<Application>()
        .SetGroupName("GridGooseSV")
        .AddConstructor<GOOSEPublisher>()
        .AddAttribute(
            "DeviceIndex",
            "Index of the NetDevice that will be used to send message. 0 by default",
            UintegerValue(0),
            MakeUintegerAccessor(&GOOSEPublisher::deviceIndex),
            MakeUintegerChecker<u_int64_t>()

        )
        .AddAttribute(
            "MaxPackets",
            "Maximum number of packets to send or 0 for inifinite",
             UintegerValue(0),
            MakeUintegerAccessor(&GOOSEPublisher::count),
            MakeUintegerChecker<u_int64_t>()
        )
        .AddAttribute(
            "AppId",
            "The application id",
            UintegerValue(1000),
            MakeUintegerAccessor(&GOOSEPublisher::appId),
            MakeUintegerChecker<uint16_t>()
        )
        .AddAttribute(
            "T0",
            "The T0 time",
            TimeValue(MilliSeconds(20)),
            MakeTimeAccessor(&GOOSEPublisher::t0),
            MakeTimeChecker()
        )
        .AddAttribute(
            "T1",
            "The T1 time",
            TimeValue(MilliSeconds(3)),
            MakeTimeAccessor(&GOOSEPublisher::t1),
            MakeTimeChecker()
        )
        .AddAttribute(
            "EventInterval",
            "Interval between events. Disable events with 0s",
            TimeValue(Seconds(0)),
            MakeTimeAccessor(&GOOSEPublisher::eventInterval),
            MakeTimeChecker()
        )
        .AddAttribute(
            "EventMessages",
            "Number of messages sent when an event occures",
            UintegerValue(5),
            MakeUintegerAccessor(&GOOSEPublisher::eventMessages),
            MakeUintegerChecker<uint8_t>()
        )
        .AddTraceSource(
            "Sent",
            "Number of sent packages",
            MakeTraceSourceAccessor(&GOOSEPublisher::sent),
            "ns3::TracedValueCallback::Uint64"
        )
    ;

    return tid;
}

void
ns3::GOOSEPublisher::StartApplication()
{
    NS_LOG_FUNCTION(this);
    auto nodeId = this->GetNode()->GetId();

    auto path = "/NodeList/" + std::to_string(nodeId) + "/DeviceList/" + std::to_string(this->deviceIndex);

    this->dataSetValues = LinkedList_create();

    LinkedList_add(dataSetValues, MmsValue_newIntegerFromInt32(1234));
    LinkedList_add(dataSetValues, MmsValue_newBinaryTime(false));
    LinkedList_add(dataSetValues, MmsValue_newIntegerFromInt32(5678));

    libiec61850::CommParameters gooseCommParameters;

    gooseCommParameters.appId = this->appId;

    this->sendEvents = this->eventInterval != Time(0);
    this->eventCount = 1;

    // TODO: this should be an attribute
    gooseCommParameters.dstAddress[0] = 0x01;
    gooseCommParameters.dstAddress[1] = 0x0c;
    gooseCommParameters.dstAddress[2] = 0xcd;
    gooseCommParameters.dstAddress[3] = 0x01;
    gooseCommParameters.dstAddress[4] = 0x00;
    gooseCommParameters.dstAddress[5] = 0x01;
    gooseCommParameters.vlanId = 0;
    gooseCommParameters.vlanPriority = 4;

    this->publisher = GoosePublisher_create(
        &gooseCommParameters,
        path.c_str()
    );

    char goCbRef[] = "simpleIOGenericIO/LLN0$GO$gcbAnalogValues";
    GoosePublisher_setGoCbRef(publisher, goCbRef);
    GoosePublisher_setConfRev(publisher, 1);

    char dataSetRef[] = "simpleIOGenericIO/LLN0$AnalogValues";
    GoosePublisher_setDataSetRef(publisher, dataSetRef);
    GoosePublisher_setTimeAllowedToLive(publisher, 500);

    this->sent = 0;
    if (this->count <= 0) this->count = -1;

    if (this->sendEvents) {
        this->gooseEventId = ns3::Simulator::Schedule(
            this->eventInterval,
            &GOOSEPublisher::SendEvent,
            this
            );
    }

    this->eventId = ns3::Simulator::ScheduleNow(
        &GOOSEPublisher::Send,
        this
        );
}

void
ns3::GOOSEPublisher::Send()
{
    if (this->count == 0) return;

    libiec61850::GoosePublisher_increaseStNum(this->publisher);

    libiec61850::GoosePublisher_publish(
        this->publisher,
        dataSetValues
    );

    this->sent++;
    this->count--;

    this->eventId = ns3::Simulator::Schedule(
        this->t0,
        &GOOSEPublisher::Send,
        this
        );
}

void ns3::GOOSEPublisher::SendEvent() {
    if (this->count == 0) return;
    Simulator::Cancel(this->eventId);

    if (this->eventCount > this->eventMessages) {
        this->eventCount = 1;

        this->gooseEventId = ns3::Simulator::Schedule(
            this->eventInterval,
            &GOOSEPublisher::SendEvent,
            this
            );

        return this->Send();
    }

    libiec61850::GoosePublisher_increaseStNum(this->publisher);

    libiec61850::GoosePublisher_publish(
        this->publisher,
        dataSetValues
    );

    this->sent++;
    this->count--;

    this->eventId = ns3::Simulator::Schedule(
        this->T(this->eventCount),
        &GOOSEPublisher::SendEvent,
        this
        );

    this->eventCount++;
}

ns3::Time ns3::GOOSEPublisher::T(const uint8_t n) {
    const auto rate = std::pow(
        this->t0.GetNanoSeconds() / this->t1.GetNanoSeconds(),
        1. / (this->eventMessages)
    );

    switch (n) {
        case 0:
            return this->t0;
        case 1:
            return this->t1;
        default:
            return NanoSeconds(this->t1.GetNanoSeconds() * pow(rate, n - 1));
    }
}

void
ns3::GOOSEPublisher::StopApplication()
{
    NS_LOG_FUNCTION(this);
    Simulator::Cancel(this->eventId);
    if (this->sendEvents) Simulator::Cancel(this->gooseEventId);
    libiec61850::GoosePublisher_destroy(this->publisher);
    LinkedList_destroyDeep(
        this->dataSetValues,
        reinterpret_cast<LinkedListValueDeleteFunction>(MmsValue_delete)
        );
}
